<?php
/**
 * Gère les interactions avec le fichier data/site.xml
 */


global $site;

$site = new Site();

/**
 * Regroupe toutes les actions relatives au fonctionnement du site
 */
class Site {
    /** Nom du site */
    private $name;
    /** Toutes les pages du site (SimpleXML) */
    private $pages;
    /** Site internet (SimpleXML) */
    private $xmlsite;

    /** Charge le site
     * 
     * Charge dans $xmlsite le XML /data/site.xml (SimpleXML)<br/>
     * Charge dans $name le nom du site défini dans /data/site.xml<br/>
     * Charge dans $pages l'ensemble des pages défini dans /data/site.xml (SimpleXML);
     * 
     * @return void
     */
    public function __construct() {
        $this -> xmlsite = simplexml_load_file($_SERVER["DOCUMENT_ROOT"]."/data/site.xml");

        $this -> pages = array();
        $this -> name = $this -> xmlsite -> name;
        $this -> pages = $this -> xmlsite -> pages -> page;
    }
    
    /**
     * Obtient le nom du site
     * @return string Nom du site
     */
    public function getSiteName()
    {
        return $this->name;
    }

    /**
     * Sauvegarde le fichier XML du site (après que des modifications aient été apportées)
     * @return void
     */
    private function saveXml() {
        $this -> xmlsite -> asXML($_SERVER["DOCUMENT_ROOT"]."/data/site.xml");
    }

    /**
     * Obtient la liste complète des pages classées par ordre d'apparition (id)
     * @return string[][] Tableau associatif de type = "id" => {nom, permalink, page d'accueil}
     */
    public function getPages() {

        if (count($this -> pages) == 0) {
            return FALSE;
        } else {
            $pages = array();
            foreach ($this->pages as $page) {
                $id = $page["id"];
                if(isset($page["home"])) {
                    $pages["$id"] = array("name" => $page -> name, "permalink" => $page -> permalink, "home" => TRUE);
                } else { 
                    $pages["$id"] = array("name" => $page -> name, "permalink" => $page -> permalink, "home" => FALSE);
                }    
                
            }
            ksort($pages);
            return $pages;
        }
    }
    
    /**
     * Définit la page d'accueil du site
     * @return void
     * @param int $id Id de la page à définir comme page d'accueil
     */
    public function setHomePage($id) {
        $xml = $this -> xmlsite -> xpath("//site/pages/page[@home]");
        unset($xml[0]["home"]);
               
        $xml = $this -> xmlsite -> xpath("//site/pages/page[@id=$id]");
        $xml[0]->addAttribute("home",1);
        $this->saveXml();
    }
    
    /**
     * Obtient la page d'accueil du site
     * @return object La page d'accueil (SimpleXML) ou FALSE si la page n'existe pas
     */
    public function getHomePage()
    {
        $xml = $this -> xmlsite -> xpath("//site/pages/page[@home]");
        
        if(count($xml)>0)
            return $xml[0];
        else
            return false;
    }
    
    /**
     * Crée une nouvelle page dans le site
     * @return void
     * @param string $name Nom de la page tel qu'il sera affiché
     * @param string $permalink Permalien de la page pour l'url
     */
    public function addPage($name, $permalink) {
        $fichier = fopen($_SERVER["DOCUMENT_ROOT"]."/template/".$permalink.".inc", 'a'); 
        fwrite($fichier,NULL);
        fclose($fichier);
        $xml = $this -> xmlsite -> pages;
        $page = $xml -> addChild("page");
        if(count($this->pages)==1)
            $page->addAttribute("home",1);
        $page -> addAttribute("uniqueId", time());
        $page -> addAttribute("id", count($this -> pages));
        $page -> addChild("name", $name);
        $page -> addChild("permalink", $permalink);
        $page -> addChild("modules");
        
        $this -> saveXml();
    }

    /**
     * Déplace une page
     * @return void
     * @param int $id Id de la page à déplacer
     * @param string $direction "up" pour déplacer la page en haut, "down" pour déplacer la page vers le bas 
     */
    public function movePage($id, $direction) {
        if ($direction == "up")
            $newid = $id - 1;
        else
            $newid = $id + 1;

        $xml = $this -> xmlsite -> xpath("//site/pages/page[@id=$id]");
        $pageOrigin = $xml[0];
        $xml = $this -> xmlsite -> xpath("//site/pages/page[@id=$newid]");
        $pageDest = $xml[0];

        $pageDest["id"] = $id;
        $pageOrigin["id"] = $newid;

        $this -> saveXml();
    }
    /**
     * Obtient une page
     * @return object La page sélectionnée en XML (SimpleXML)
     * @param int $id Id de la page à sélectionner
     */
    public function getPage($id) {
        if(is_numeric($id))
            $xml = $this -> xmlsite -> xpath("//site/pages/page[@id=$id]");
        else            
            $xml = $this -> xmlsite -> xpath("//site/pages/page[permalink='$id']");
                    
        return $xml[0];
    }
    
    /**
     * Obtient les modules installés sur une page
     * @return object L'ensemble des modules de la page sous forme de tableau XML (SimpleXML).
     * @param int $idpage Id de la page
     * @example Pour obtenir le premier module :
     * <blockquote>$var = $site->getModules($idpage);
     * $module1 = $var[0];</blockquote>
     */
    public function getModules($idpage) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules/module");
        if (count($modules) == 0) {
            return FALSE;
        } else {
            return $modules;
        }

    }
    
    /**
     * Obtient les modules généraux installés sur le site
     * @return object L'ensemble des modules généraux du site sous forme de tableau XML (SimpleXML).
     * @example Pour obtenir le premier module :
     * <blockquote>$var = $site->getGeneralModules();
     * $module1 = $var[0];</blockquote>
     */
    public function getGeneralModules() {
        $modules = $this -> xmlsite -> xpath("//site/modules/module");
        if (count($modules) == 0) {
            return FALSE;
        } else {
            return $modules;
        }

    }
    
    public function getGeneralModule($name) {
        $modules = $this -> xmlsite -> xpath("//site/modules/module[name='$name']");
        
        if(count($modules)>0)
            return $modules[0];
        else            
            return false;
    }
    
    public function addGeneralModule($name, $type, $short, $description) {
        $modules = $this -> xmlsite -> xpath("//site/modules");
        $mods = $modules[0];
        $newmodule = $mods -> addChild("module");
        $newmodule -> addChild("name", $name);
        $newmodule -> addChild("type", $type);
        $newmodule -> addChild("short", $short);
        $newmodule -> addChild("description", $description);
        $newmodule -> addChild("visible", 1);
        $newmodule -> addChild("params");
        Actualis::createModule($newmodule);
        $this -> saveXml();
    }
    
    public function updateGeneralModule($name, $newname, $type, $short, $description) {
        $modules = $this -> xmlsite -> xpath("//site/modules/module[name='$name']");
        $module = $modules[0];
        
        $module -> name = $newname;
        $module -> short = $short;
        $module -> description = $description;
        
        if($module -> type != $type)
        {
            $params = $module -> params;
            $module -> type = $type;
            unset($params[0]); 
            $module -> addChild("params");
        }
        
        $this->saveXml();
    }
    
    public function saveGeneralModule($name, $post, $files, $get) {
        $modules = $this -> xmlsite -> xpath("//site/modules/module[name='$name']");
        $module = $modules[0];
        
        Actualis::saveModule($module,$post, $files, $get);
        $this -> saveXml();
    }
    
    
    public function deleteGeneralModule($name) {
        $modules = $this -> xmlsite -> xpath("//site/modules/module[name='$name']");
        $module = $modules[0];
        unset($module[0]);
        $this -> saveXml();
    }
    

    public function updatePage($id, $name, $permalink) {
        $xml = $this -> xmlsite -> xpath("//site/pages/page[@id=$id]");
        $page = $xml[0];
        $page -> name = $name;
        if ($page->permalink != $permalink) {
            rename($_SERVER["DOCUMENT_ROOT"]."/template/".$page->permalink.".inc",$_SERVER["DOCUMENT_ROOT"]."/template/".$permalink.".inc");
        }
        $page -> permalink = $permalink;

        $this -> saveXml();
    }

    public function addModule($idpage, $name, $type, $short, $description) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules");
        $mods = $modules[0];
        $newmodule = $mods -> addChild("module");
        $newmodule -> addChild("name", $name);
        $newmodule -> addChild("type", $type);
        $newmodule -> addChild("short", $short);
        $newmodule -> addChild("description", $description);
        $newmodule -> addChild("visible", 1);
        $newmodule -> addChild("params");
        Actualis::createModule($newmodule);
        $this -> saveXml();
    }

    public function deleteModule($idpage, $name) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules/module");
        foreach ($modules as $mod) {
            if ($mod -> name == $name) {
                Actualis::deleteModule($mod);
                unset($mod[0]);
                $this -> saveXml();
            }
        }
    }
    
    public function saveModule($idpage, $name, $post, $files, $get) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules/module");
        $module;
        foreach ($modules as $mod) {
            if ($mod -> name == $name) {
                $module = $mod[0];
            }
        }
        Actualis::saveModule($module,$post, $files, $get);
        $this -> saveXml();
    }
    
    public function getModule($idpage, $name) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules/module[name='$name']");
        
        if(count($modules)>0)
            return $modules[0];
        else            
            return false;
        }
    
    public function deletePage($idpage) {
        $page = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]");
        unlink($_SERVER["DOCUMENT_ROOT"]."/template/".$page[0]->permalink.".inc");
        unset($page[0][0]);
        
        $pages = $this -> xmlsite -> pages -> page;
        foreach($pages as $p)
        {
            if($p["id"] >= $idpage) {
                $p["id"] = intval($p["id"]-1);
            }
        }
        
        $this -> saveXml();       
    }
    
    public function updateModule($idpage, $name, $newname, $type, $short, $description) {
        $modules = $this -> xmlsite -> xpath("//site/pages/page[@id=$idpage]/modules/module");
        $module;
        foreach ($modules as $mod) {
            if ($mod -> name == $name) {
                $module = $mod[0];
            }
        }
        
        $module -> name = $newname;
        $module -> short = $short;
        $module -> description = $description;
        
        if($module -> type != $type)
        {
            $params = $module -> params;
            $module -> type = $type;
            unset($params[0]); 
            $module -> addChild("params");
        }
        
        $this->saveXml();
    }
    
    public function toggleModuleVisibility($module)
    {
        $visible = $module -> visible;
        if($visible == 1)
            $module -> visible = 0;
        else
            $module -> visible = 1;
        
        $this->saveXml();
    }
    
    public function getXml()
    {
        return $this -> xmlsite;
    }
    

}
?>